/*
 * wintrust softpub functions tests
 *
 * Copyright 2007,2010 Juan Lang
 * Copyright 2010 Andrey Turkin
 * Copyright 2016 Mark Jansen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <stdio.h>
#include <stdarg.h>

#include <windef.h>
#include <winbase.h>
#include <winerror.h>
#include <wintrust.h>
#include <softpub.h>
#include <mssip.h>
#include <winuser.h>
#include "winnls.h"

#include "wine/test.h"

/* Just in case we're being built with borked headers, redefine function
 * pointers to have the correct calling convention.
 */
typedef void   *(WINAPI *SAFE_MEM_ALLOC)(DWORD);
typedef void    (WINAPI *SAFE_MEM_FREE)(void *);
typedef BOOL    (WINAPI *SAFE_ADD_STORE)(CRYPT_PROVIDER_DATA *,
 HCERTSTORE);
typedef BOOL    (WINAPI *SAFE_ADD_SGNR)(CRYPT_PROVIDER_DATA *,
 BOOL, DWORD, struct _CRYPT_PROVIDER_SGNR *);
typedef BOOL    (WINAPI *SAFE_ADD_CERT)(CRYPT_PROVIDER_DATA *,
 DWORD, BOOL, DWORD, PCCERT_CONTEXT);
typedef BOOL    (WINAPI *SAFE_ADD_PRIVDATA)(CRYPT_PROVIDER_DATA *,
 CRYPT_PROVIDER_PRIVDATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_INIT_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_OBJTRUST_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_SIGTRUST_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_CERTTRUST_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_FINALPOLICY_CALL)(CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_TESTFINALPOLICY_CALL)(
 CRYPT_PROVIDER_DATA *);
typedef HRESULT (WINAPI *SAFE_PROVIDER_CLEANUP_CALL)(CRYPT_PROVIDER_DATA *);
typedef BOOL    (WINAPI *SAFE_PROVIDER_CERTCHKPOLICY_CALL)(
 CRYPT_PROVIDER_DATA *, DWORD, BOOL, DWORD);

typedef struct _SAFE_PROVIDER_FUNCTIONS
{
    DWORD                              cbStruct;
    SAFE_MEM_ALLOC                     pfnAlloc;
    SAFE_MEM_FREE                      pfnFree;
    SAFE_ADD_STORE                     pfnAddStore2Chain;
    SAFE_ADD_SGNR                      pfnAddSgnr2Chain;
    SAFE_ADD_CERT                      pfnAddCert2Chain;
    SAFE_ADD_PRIVDATA                  pfnAddPrivData2Chain;
    SAFE_PROVIDER_INIT_CALL            pfnInitialize;
    SAFE_PROVIDER_OBJTRUST_CALL        pfnObjectTrust;
    SAFE_PROVIDER_SIGTRUST_CALL        pfnSignatureTrust;
    SAFE_PROVIDER_CERTTRUST_CALL       pfnCertificateTrust;
    SAFE_PROVIDER_FINALPOLICY_CALL     pfnFinalPolicy;
    SAFE_PROVIDER_CERTCHKPOLICY_CALL   pfnCertCheckPolicy;
    SAFE_PROVIDER_TESTFINALPOLICY_CALL pfnTestFinalPolicy;
    struct _CRYPT_PROVUI_FUNCS        *psUIpfns;
    SAFE_PROVIDER_CLEANUP_CALL         pfnCleanupPolicy;
} SAFE_PROVIDER_FUNCTIONS;

static BOOL (WINAPI * pWTHelperGetKnownUsages)(DWORD action, PCCRYPT_OID_INFO **usages);
static BOOL (WINAPI * CryptSIPCreateIndirectData_p)(SIP_SUBJECTINFO *, DWORD *, SIP_INDIRECT_DATA *);
static VOID (WINAPI * CertFreeCertificateChain_p)(PCCERT_CHAIN_CONTEXT);

static void InitFunctionPtrs(void)
{
    HMODULE hWintrust = GetModuleHandleA("wintrust.dll");
    HMODULE hCrypt32 = GetModuleHandleA("crypt32.dll");

#define WINTRUST_GET_PROC(func) \
    p ## func = (void*)GetProcAddress(hWintrust, #func); \
    if(!p ## func) { \
      trace("GetProcAddress(%s) failed\n", #func); \
    }

    WINTRUST_GET_PROC(WTHelperGetKnownUsages)

#undef WINTRUST_GET_PROC

#define CRYPT32_GET_PROC(func) \
    func ## _p = (void*)GetProcAddress(hCrypt32, #func); \
    if(!func ## _p) { \
      trace("GetProcAddress(%s) failed\n", #func); \
    }

    CRYPT32_GET_PROC(CryptSIPCreateIndirectData)
    CRYPT32_GET_PROC(CertFreeCertificateChain)

#undef CRYPT32_GET_PROC
}

static const BYTE v1CertWithPubKey[] = {
0x30,0x81,0x95,0x02,0x01,0x01,0x30,0x02,0x06,0x00,0x30,0x15,0x31,0x13,0x30,
0x11,0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x4a,0x75,0x61,0x6e,0x20,0x4c,0x61,
0x6e,0x67,0x00,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x30,0x31,0x30,0x31,
0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x18,0x0f,0x31,0x36,0x30,0x31,0x30,0x31,
0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x15,0x31,0x13,0x30,0x11,
0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x4a,0x75,0x61,0x6e,0x20,0x4c,0x61,0x6e,
0x67,0x00,0x30,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x01,0x05,0x00,0x03,0x11,0x00,0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0xa3,0x16,0x30,0x14,0x30,0x12,0x06,
0x03,0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x08,0x30,0x06,0x01,0x01,0xff,0x02,
0x01,0x01 };

static void test_utils(SAFE_PROVIDER_FUNCTIONS *funcs)
{
    CRYPT_PROVIDER_DATA data = { 0 };
    HCERTSTORE store;
    CRYPT_PROVIDER_SGNR sgnr = { 0 };
    BOOL ret;

    /* Crash
    ret = funcs->pfnAddStore2Chain(NULL, NULL);
    ret = funcs->pfnAddStore2Chain(&data, NULL);
     */
    store = CertOpenStore(CERT_STORE_PROV_MEMORY, X509_ASN_ENCODING, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    if (store)
    {
        ret = funcs->pfnAddStore2Chain(&data, store);
        ok(ret, "pfnAddStore2Chain failed: %08x\n", GetLastError());
        ok(data.chStores == 1, "Expected 1 store, got %d\n", data.chStores);
        ok(data.pahStores != NULL, "Expected pahStores to be allocated\n");
        if (data.pahStores)
        {
            ok(data.pahStores[0] == store, "Unexpected store\n");
            CertCloseStore(data.pahStores[0], 0);
            funcs->pfnFree(data.pahStores);
            data.pahStores = NULL;
            data.chStores = 0;
            CertCloseStore(store, 0);
            store = NULL;
        }
    }
    else
        skip("CertOpenStore failed: %08x\n", GetLastError());

    /* Crash
    ret = funcs->pfnAddSgnr2Chain(NULL, FALSE, 0, NULL);
    ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, NULL);
     */
    ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, &sgnr);
    ok(ret, "pfnAddSgnr2Chain failed: %08x\n", GetLastError());
    ok(data.csSigners == 1, "Expected 1 signer, got %d\n", data.csSigners);
    ok(data.pasSigners != NULL, "Expected pasSigners to be allocated\n");
    if (data.pasSigners)
    {
        PCCERT_CONTEXT cert;

        ok(!memcmp(&data.pasSigners[0], &sgnr, sizeof(sgnr)),
         "Unexpected data in signer\n");
        /* Adds into the location specified by the index */
        sgnr.cbStruct = sizeof(CRYPT_PROVIDER_SGNR);
        sgnr.sftVerifyAsOf.dwLowDateTime = 0xdeadbeef;
        ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 1, &sgnr);
        ok(ret, "pfnAddSgnr2Chain failed: %08x\n", GetLastError());
        ok(data.csSigners == 2, "Expected 2 signers, got %d\n", data.csSigners);
        ok(!memcmp(&data.pasSigners[1], &sgnr, sizeof(sgnr)),
         "Unexpected data in signer\n");
        /* This also adds, but the index is ignored */
        sgnr.cbStruct = sizeof(DWORD);
        ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, &sgnr);
        ok(ret, "pfnAddSgnr2Chain failed: %08x\n", GetLastError());
        ok(data.csSigners == 3, "Expected 3 signers, got %d\n", data.csSigners);
        sgnr.sftVerifyAsOf.dwLowDateTime = 0;
        todo_wine
        ok(!memcmp(&data.pasSigners[2], &sgnr, sizeof(sgnr)),
           "Unexpected data in signer\n");
        /* But too large a thing isn't added */
        sgnr.cbStruct = sizeof(sgnr) + sizeof(DWORD);
        SetLastError(0xdeadbeef);
        ret = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, &sgnr);
        ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
         "Expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());

        /* Crash
        ret = funcs->pfnAddCert2Chain(NULL, 0, FALSE, 0, NULL);
        ret = funcs->pfnAddCert2Chain(&data, 0, FALSE, 0, NULL);
         */
        cert = CertCreateCertificateContext(X509_ASN_ENCODING, v1CertWithPubKey,
         sizeof(v1CertWithPubKey));
        if (cert)
        {
            /* Notes on behavior that are hard to test:
             * 1. If pasSigners is invalid, pfnAddCert2Chain crashes
             * 2. An invalid signer index isn't checked.
             */
            ret = funcs->pfnAddCert2Chain(&data, 0, FALSE, 0, cert);
            ok(ret, "pfnAddCert2Chain failed: %08x\n", GetLastError());
            ok(data.pasSigners[0].csCertChain == 1, "Expected 1 cert, got %d\n",
             data.pasSigners[0].csCertChain);
            ok(data.pasSigners[0].pasCertChain != NULL,
             "Expected pasCertChain to be allocated\n");
            if (data.pasSigners[0].pasCertChain)
            {
                ok(data.pasSigners[0].pasCertChain[0].pCert == cert,
                 "Unexpected cert\n");
                CertFreeCertificateContext(
                 data.pasSigners[0].pasCertChain[0].pCert);
            }
            CertFreeCertificateContext(cert);
        }
        else
            skip("CertCreateCertificateContext failed: %08x\n", GetLastError());
        funcs->pfnFree(data.pasSigners);
    }
}

static void testInitialize(SAFE_PROVIDER_FUNCTIONS *funcs, GUID *actionID)
{
    HRESULT ret;
    CRYPT_PROVIDER_DATA data = { 0 };
    WINTRUST_DATA wintrust_data = { 0 };

    if (!funcs->pfnInitialize)
    {
        skip("missing pfnInitialize\n");
        return;
    }

    /* Crashes
    ret = funcs->pfnInitialize(NULL);
     */
    memset(&data, 0, sizeof(data));
    ret = funcs->pfnInitialize(&data);
    ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
    data.padwTrustStepErrors =
     funcs->pfnAlloc(TRUSTERROR_MAX_STEPS * sizeof(DWORD));
    /* Without wintrust data set, crashes when padwTrustStepErrors is set */
    data.pWintrustData = &wintrust_data;
    if (data.padwTrustStepErrors)
    {
        /* Apparently, cdwTrustStepErrors does not need to be set. */
        memset(data.padwTrustStepErrors, 0,
         TRUSTERROR_MAX_STEPS * sizeof(DWORD));
        ret = funcs->pfnInitialize(&data);
        ok(ret == S_OK, "Expected S_OK, got %08x\n", ret);
        data.cdwTrustStepErrors = 1;
        ret = funcs->pfnInitialize(&data);
        ok(ret == S_OK, "Expected S_OK, got %08x\n", ret);
        memset(data.padwTrustStepErrors, 0xba,
         TRUSTERROR_MAX_STEPS * sizeof(DWORD));
        ret = funcs->pfnInitialize(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
        data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_WVTINIT] = 0;
        ret = funcs->pfnInitialize(&data);
        ok(ret == S_OK, "Expected S_OK, got %08x\n", ret);
        funcs->pfnFree(data.padwTrustStepErrors);
    }
}

static void getNotepadPath(WCHAR *notepadPathW, DWORD size)
{
    static const CHAR notepad[] = "\\notepad.exe";
    CHAR notepadPath[MAX_PATH];

    /* Workaround missing W-functions for win9x */
    GetWindowsDirectoryA(notepadPath, MAX_PATH);
    lstrcatA(notepadPath, notepad);
    MultiByteToWideChar(CP_ACP, 0, notepadPath, -1, notepadPathW, size);
}

/* Creates a test file and returns a handle to it.  The file's path is returned
 * in temp_file, which must be at least MAX_PATH characters in length.
 */
static HANDLE create_temp_file(WCHAR *temp_file)
{
    HANDLE file = INVALID_HANDLE_VALUE;
    WCHAR temp_path[MAX_PATH];

    if (GetTempPathW(sizeof(temp_path) / sizeof(temp_path[0]), temp_path))
    {
        static const WCHAR img[] = { 'i','m','g',0 };

        if (GetTempFileNameW(temp_path, img, 0, temp_file))
            file = CreateFileW(temp_file, GENERIC_READ | GENERIC_WRITE, 0, NULL,
             CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
    }
    return file;
}

static void testObjTrust(SAFE_PROVIDER_FUNCTIONS *funcs, GUID *actionID)
{
    HRESULT ret;
    CRYPT_PROVIDER_DATA data = { 0 };
    CRYPT_PROVIDER_SIGSTATE sig_state = { 0 };
    WINTRUST_DATA wintrust_data = { 0 };
    WINTRUST_CERT_INFO certInfo = { sizeof(WINTRUST_CERT_INFO), 0 };
    WINTRUST_FILE_INFO fileInfo = { sizeof(WINTRUST_FILE_INFO), 0 };

    if (!funcs->pfnObjectTrust)
    {
        skip("missing pfnObjectTrust\n");
        return;
    }

    /* Crashes
    ret = funcs->pfnObjectTrust(NULL);
     */
    data.pSigState = &sig_state;
    data.pWintrustData = &wintrust_data;
    data.padwTrustStepErrors =
     funcs->pfnAlloc(TRUSTERROR_MAX_STEPS * sizeof(DWORD));
    if (data.padwTrustStepErrors)
    {
        WCHAR pathW[MAX_PATH];
        PROVDATA_SIP provDataSIP = { 0 };
        static const GUID unknown = { 0xC689AAB8, 0x8E78, 0x11D0, { 0x8C,0x47,
         0x00,0xC0,0x4F,0xC2,0x95,0xEE } };
        static GUID bogusGuid = { 0xdeadbeef, 0xbaad, 0xf00d, { 0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00 } };

        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         ERROR_INVALID_PARAMETER,
         "Expected ERROR_INVALID_PARAMETER, got %08x\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        U(wintrust_data).pCert = &certInfo;
        wintrust_data.dwUnionChoice = WTD_CHOICE_CERT;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_OK, "Expected S_OK, got %08x\n", ret);
        certInfo.psCertContext = (PCERT_CONTEXT)CertCreateCertificateContext(
         X509_ASN_ENCODING, v1CertWithPubKey, sizeof(v1CertWithPubKey));
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_OK, "Expected S_OK, got %08x\n", ret);
        CertFreeCertificateContext(certInfo.psCertContext);
        certInfo.psCertContext = NULL;
        wintrust_data.dwUnionChoice = WTD_CHOICE_FILE;
        U(wintrust_data).pFile = NULL;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         ERROR_INVALID_PARAMETER,
         "Expected ERROR_INVALID_PARAMETER, got %08x\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        U(wintrust_data).pFile = &fileInfo;
        /* Crashes
        ret = funcs->pfnObjectTrust(&data);
         */
        /* Create and test with an empty file */
        fileInfo.hFile = create_temp_file(pathW);
        /* pfnObjectTrust now crashes unless both pPDSip and psPfns are set */
        U(data).pPDSip = &provDataSIP;
        data.psPfns = (CRYPT_PROVIDER_FUNCTIONS *)funcs;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_SUBJECT_FORM_UNKNOWN,
         "expected TRUST_E_SUBJECT_FORM_UNKNOWN, got %08x\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        CloseHandle(fileInfo.hFile);
        fileInfo.hFile = NULL;
        fileInfo.pcwszFilePath = pathW;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_SUBJECT_FORM_UNKNOWN,
         "expected TRUST_E_SUBJECT_FORM_UNKNOWN, got %08x\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        DeleteFileW(pathW);
        /* Test again with a file we expect to exist, and to contain no
         * signature.
         */
        getNotepadPath(pathW, MAX_PATH);
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_NOSIGNATURE ||
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_SUBJECT_FORM_UNKNOWN,
         "Expected TRUST_E_NOSIGNATURE or TRUST_E_SUBJECT_FORM_UNKNOWN, got %08x\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        if (data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_NOSIGNATURE)
        {
            ok(!memcmp(&provDataSIP.gSubject, &unknown, sizeof(unknown)),
             "Unexpected subject GUID\n");
            ok(provDataSIP.pSip != NULL, "Expected a SIP\n");
            ok(provDataSIP.psSipSubjectInfo != NULL,
             "Expected a subject info\n");
        }
        /* Specifying the GUID results in that GUID being the subject GUID */
        fileInfo.pgKnownSubject = &bogusGuid;
        ret = funcs->pfnObjectTrust(&data);
        ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
        ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_NOSIGNATURE ||
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_SUBJECT_FORM_UNKNOWN ||
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_PROVIDER_UNKNOWN,
         "Expected TRUST_E_NOSIGNATURE or TRUST_E_SUBJECT_FORM_UNKNOWN or TRUST_E_PROVIDER_UNKNOWN, got %08x\n",
         data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV]);
        if (data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_OBJPROV] ==
         TRUST_E_NOSIGNATURE)
        {
            ok(!memcmp(&provDataSIP.gSubject, &bogusGuid, sizeof(bogusGuid)),
             "unexpected subject GUID\n");
        }
        /* Specifying a bogus GUID pointer crashes */
        if (0)
        {
            fileInfo.pgKnownSubject = (GUID *)0xdeadbeef;
            ret = funcs->pfnObjectTrust(&data);
            ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
        }
        funcs->pfnFree(data.padwTrustStepErrors);
    }
}

static const BYTE selfSignedCert[] = {
  0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x43,
  0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d,
  0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x44, 0x70, 0x7a, 0x43, 0x43,
  0x41, 0x6f, 0x2b, 0x67, 0x41, 0x77, 0x49, 0x42, 0x41, 0x67, 0x49, 0x4a,
  0x41, 0x4c, 0x59, 0x51, 0x67, 0x65, 0x66, 0x7a, 0x51, 0x41, 0x61, 0x43,
  0x4d, 0x41, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33,
  0x44, 0x51, 0x45, 0x42, 0x42, 0x51, 0x55, 0x41, 0x4d, 0x47, 0x6f, 0x78,
  0x43, 0x7a, 0x41, 0x4a, 0x42, 0x67, 0x4e, 0x56, 0x0a, 0x42, 0x41, 0x59,
  0x54, 0x41, 0x6b, 0x46, 0x56, 0x4d, 0x52, 0x4d, 0x77, 0x45, 0x51, 0x59,
  0x44, 0x56, 0x51, 0x51, 0x49, 0x44, 0x41, 0x70, 0x54, 0x62, 0x32, 0x31,
  0x6c, 0x4c, 0x56, 0x4e, 0x30, 0x59, 0x58, 0x52, 0x6c, 0x4d, 0x53, 0x45,
  0x77, 0x48, 0x77, 0x59, 0x44, 0x56, 0x51, 0x51, 0x4b, 0x44, 0x42, 0x68,
  0x4a, 0x62, 0x6e, 0x52, 0x6c, 0x63, 0x6d, 0x35, 0x6c, 0x64, 0x43, 0x42,
  0x58, 0x0a, 0x61, 0x57, 0x52, 0x6e, 0x61, 0x58, 0x52, 0x7a, 0x49, 0x46,
  0x42, 0x30, 0x65, 0x53, 0x42, 0x4d, 0x64, 0x47, 0x51, 0x78, 0x49, 0x7a,
  0x41, 0x68, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x4d, 0x4d, 0x47, 0x6e,
  0x4e, 0x6c, 0x62, 0x47, 0x5a, 0x7a, 0x61, 0x57, 0x64, 0x75, 0x5a, 0x57,
  0x51, 0x75, 0x64, 0x47, 0x56, 0x7a, 0x64, 0x43, 0x35, 0x33, 0x61, 0x57,
  0x35, 0x6c, 0x61, 0x48, 0x45, 0x75, 0x0a, 0x62, 0x33, 0x4a, 0x6e, 0x4d,
  0x42, 0x34, 0x58, 0x44, 0x54, 0x45, 0x7a, 0x4d, 0x44, 0x59, 0x79, 0x4d,
  0x54, 0x45, 0x78, 0x4d, 0x6a, 0x55, 0x78, 0x4d, 0x46, 0x6f, 0x58, 0x44,
  0x54, 0x49, 0x7a, 0x4d, 0x44, 0x59, 0x78, 0x4f, 0x54, 0x45, 0x78, 0x4d,
  0x6a, 0x55, 0x78, 0x4d, 0x46, 0x6f, 0x77, 0x61, 0x6a, 0x45, 0x4c, 0x4d,
  0x41, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x68, 0x4d, 0x43, 0x0a,
  0x51, 0x56, 0x55, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42, 0x67, 0x4e, 0x56,
  0x42, 0x41, 0x67, 0x4d, 0x43, 0x6c, 0x4e, 0x76, 0x62, 0x57, 0x55, 0x74,
  0x55, 0x33, 0x52, 0x68, 0x64, 0x47, 0x55, 0x78, 0x49, 0x54, 0x41, 0x66,
  0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x6f, 0x4d, 0x47, 0x45, 0x6c, 0x75,
  0x64, 0x47, 0x56, 0x79, 0x62, 0x6d, 0x56, 0x30, 0x49, 0x46, 0x64, 0x70,
  0x5a, 0x47, 0x64, 0x70, 0x0a, 0x64, 0x48, 0x4d, 0x67, 0x55, 0x48, 0x52,
  0x35, 0x49, 0x45, 0x78, 0x30, 0x5a, 0x44, 0x45, 0x6a, 0x4d, 0x43, 0x45,
  0x47, 0x41, 0x31, 0x55, 0x45, 0x41, 0x77, 0x77, 0x61, 0x63, 0x32, 0x56,
  0x73, 0x5a, 0x6e, 0x4e, 0x70, 0x5a, 0x32, 0x35, 0x6c, 0x5a, 0x43, 0x35,
  0x30, 0x5a, 0x58, 0x4e, 0x30, 0x4c, 0x6e, 0x64, 0x70, 0x62, 0x6d, 0x56,
  0x6f, 0x63, 0x53, 0x35, 0x76, 0x63, 0x6d, 0x63, 0x77, 0x0a, 0x67, 0x67,
  0x45, 0x69, 0x4d, 0x41, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49,
  0x62, 0x33, 0x44, 0x51, 0x45, 0x42, 0x41, 0x51, 0x55, 0x41, 0x41, 0x34,
  0x49, 0x42, 0x44, 0x77, 0x41, 0x77, 0x67, 0x67, 0x45, 0x4b, 0x41, 0x6f,
  0x49, 0x42, 0x41, 0x51, 0x44, 0x77, 0x4e, 0x6d, 0x2b, 0x46, 0x7a, 0x78,
  0x6e, 0x6b, 0x48, 0x57, 0x2f, 0x4e, 0x70, 0x37, 0x59, 0x48, 0x34, 0x4d,
  0x79, 0x45, 0x0a, 0x77, 0x4d, 0x6c, 0x49, 0x67, 0x71, 0x30, 0x66, 0x45,
  0x77, 0x70, 0x47, 0x6f, 0x41, 0x75, 0x78, 0x44, 0x64, 0x61, 0x46, 0x55,
  0x32, 0x6f, 0x70, 0x76, 0x41, 0x51, 0x56, 0x61, 0x2b, 0x41, 0x43, 0x46,
  0x38, 0x63, 0x6f, 0x38, 0x4d, 0x4a, 0x6c, 0x33, 0x78, 0x77, 0x76, 0x46,
  0x44, 0x2b, 0x67, 0x61, 0x46, 0x45, 0x7a, 0x59, 0x78, 0x53, 0x58, 0x30,
  0x43, 0x47, 0x72, 0x4a, 0x45, 0x4c, 0x63, 0x0a, 0x74, 0x34, 0x4d, 0x69,
  0x30, 0x68, 0x4b, 0x50, 0x76, 0x42, 0x70, 0x65, 0x73, 0x59, 0x6c, 0x46,
  0x4d, 0x51, 0x65, 0x6b, 0x2b, 0x63, 0x70, 0x51, 0x50, 0x33, 0x4b, 0x35,
  0x75, 0x36, 0x71, 0x58, 0x5a, 0x52, 0x49, 0x67, 0x48, 0x75, 0x59, 0x45,
  0x4c, 0x2f, 0x73, 0x55, 0x6f, 0x39, 0x32, 0x70, 0x44, 0x30, 0x7a, 0x4a,
  0x65, 0x4c, 0x47, 0x41, 0x31, 0x49, 0x30, 0x4b, 0x5a, 0x34, 0x73, 0x2f,
  0x0a, 0x51, 0x7a, 0x77, 0x61, 0x4f, 0x38, 0x62, 0x62, 0x4b, 0x6d, 0x37,
  0x42, 0x72, 0x6e, 0x56, 0x77, 0x30, 0x6e, 0x5a, 0x2f, 0x4b, 0x41, 0x5a,
  0x6a, 0x75, 0x78, 0x75, 0x6f, 0x4e, 0x33, 0x52, 0x64, 0x72, 0x69, 0x30,
  0x4a, 0x48, 0x77, 0x7a, 0x6a, 0x41, 0x55, 0x34, 0x2b, 0x71, 0x57, 0x65,
  0x55, 0x63, 0x2f, 0x64, 0x33, 0x45, 0x70, 0x4f, 0x47, 0x78, 0x69, 0x42,
  0x77, 0x5a, 0x4e, 0x61, 0x7a, 0x0a, 0x39, 0x6f, 0x4a, 0x41, 0x37, 0x54,
  0x2f, 0x51, 0x6f, 0x62, 0x75, 0x61, 0x4e, 0x53, 0x6b, 0x65, 0x55, 0x48,
  0x43, 0x61, 0x50, 0x53, 0x6a, 0x44, 0x37, 0x71, 0x7a, 0x6c, 0x43, 0x4f,
  0x52, 0x48, 0x47, 0x68, 0x75, 0x31, 0x76, 0x79, 0x79, 0x35, 0x31, 0x45,
  0x36, 0x79, 0x46, 0x43, 0x4e, 0x47, 0x66, 0x65, 0x7a, 0x71, 0x2f, 0x4d,
  0x59, 0x34, 0x4e, 0x4b, 0x68, 0x77, 0x72, 0x61, 0x59, 0x64, 0x0a, 0x62,
  0x79, 0x49, 0x2f, 0x6c, 0x42, 0x46, 0x62, 0x36, 0x35, 0x6b, 0x5a, 0x45,
  0x66, 0x49, 0x4b, 0x4b, 0x54, 0x7a, 0x79, 0x36, 0x76, 0x30, 0x44, 0x65,
  0x79, 0x50, 0x37, 0x52, 0x6b, 0x34, 0x75, 0x48, 0x44, 0x38, 0x77, 0x62,
  0x49, 0x79, 0x50, 0x32, 0x47, 0x6c, 0x42, 0x30, 0x67, 0x37, 0x2f, 0x69,
  0x79, 0x33, 0x4c, 0x61, 0x74, 0x49, 0x74, 0x49, 0x70, 0x2b, 0x49, 0x35,
  0x53, 0x50, 0x56, 0x0a, 0x41, 0x67, 0x4d, 0x42, 0x41, 0x41, 0x47, 0x6a,
  0x55, 0x44, 0x42, 0x4f, 0x4d, 0x42, 0x30, 0x47, 0x41, 0x31, 0x55, 0x64,
  0x44, 0x67, 0x51, 0x57, 0x42, 0x42, 0x53, 0x36, 0x49, 0x4c, 0x5a, 0x2f,
  0x71, 0x38, 0x66, 0x2f, 0x4b, 0x45, 0x68, 0x4b, 0x76, 0x68, 0x69, 0x2b,
  0x73, 0x6b, 0x59, 0x45, 0x31, 0x79, 0x48, 0x71, 0x39, 0x7a, 0x41, 0x66,
  0x42, 0x67, 0x4e, 0x56, 0x48, 0x53, 0x4d, 0x45, 0x0a, 0x47, 0x44, 0x41,
  0x57, 0x67, 0x42, 0x53, 0x36, 0x49, 0x4c, 0x5a, 0x2f, 0x71, 0x38, 0x66,
  0x2f, 0x4b, 0x45, 0x68, 0x4b, 0x76, 0x68, 0x69, 0x2b, 0x73, 0x6b, 0x59,
  0x45, 0x31, 0x79, 0x48, 0x71, 0x39, 0x7a, 0x41, 0x4d, 0x42, 0x67, 0x4e,
  0x56, 0x48, 0x52, 0x4d, 0x45, 0x42, 0x54, 0x41, 0x44, 0x41, 0x51, 0x48,
  0x2f, 0x4d, 0x41, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62,
  0x33, 0x0a, 0x44, 0x51, 0x45, 0x42, 0x42, 0x51, 0x55, 0x41, 0x41, 0x34,
  0x49, 0x42, 0x41, 0x51, 0x41, 0x79, 0x5a, 0x59, 0x77, 0x47, 0x4b, 0x46,
  0x34, 0x34, 0x43, 0x68, 0x47, 0x51, 0x72, 0x6e, 0x74, 0x57, 0x6c, 0x38,
  0x48, 0x53, 0x4a, 0x30, 0x63, 0x69, 0x55, 0x58, 0x4d, 0x44, 0x4b, 0x32,
  0x46, 0x6c, 0x6f, 0x74, 0x47, 0x49, 0x6a, 0x30, 0x32, 0x6c, 0x4d, 0x39,
  0x38, 0x71, 0x45, 0x49, 0x65, 0x68, 0x0a, 0x56, 0x67, 0x66, 0x41, 0x34,
  0x7a, 0x69, 0x37, 0x4d, 0x45, 0x6c, 0x51, 0x61, 0x76, 0x6b, 0x52, 0x76,
  0x32, 0x54, 0x43, 0x50, 0x50, 0x55, 0x51, 0x62, 0x35, 0x51, 0x64, 0x61,
  0x6f, 0x37, 0x57, 0x78, 0x37, 0x6c, 0x66, 0x61, 0x54, 0x6f, 0x5a, 0x68,
  0x4f, 0x54, 0x2b, 0x4e, 0x52, 0x68, 0x32, 0x6b, 0x35, 0x78, 0x2b, 0x6b,
  0x6a, 0x5a, 0x46, 0x77, 0x38, 0x70, 0x45, 0x48, 0x74, 0x35, 0x51, 0x0a,
  0x69, 0x68, 0x62, 0x46, 0x4c, 0x35, 0x58, 0x2b, 0x57, 0x7a, 0x6f, 0x2b,
  0x42, 0x36, 0x36, 0x59, 0x79, 0x49, 0x76, 0x68, 0x77, 0x54, 0x63, 0x48,
  0x30, 0x46, 0x2b, 0x6e, 0x66, 0x55, 0x71, 0x66, 0x74, 0x38, 0x59, 0x74,
  0x72, 0x2f, 0x38, 0x37, 0x47, 0x45, 0x62, 0x73, 0x41, 0x48, 0x6a, 0x48,
  0x43, 0x36, 0x4c, 0x2b, 0x77, 0x6b, 0x31, 0x76, 0x4e, 0x6e, 0x64, 0x49,
  0x59, 0x47, 0x30, 0x51, 0x0a, 0x79, 0x62, 0x73, 0x7a, 0x78, 0x49, 0x72,
  0x32, 0x6d, 0x46, 0x45, 0x49, 0x4a, 0x6f, 0x69, 0x51, 0x44, 0x44, 0x67,
  0x66, 0x6c, 0x71, 0x67, 0x64, 0x76, 0x4c, 0x54, 0x32, 0x79, 0x64, 0x46,
  0x6d, 0x79, 0x33, 0x73, 0x32, 0x68, 0x49, 0x74, 0x51, 0x6c, 0x49, 0x71,
  0x4b, 0x4c, 0x42, 0x36, 0x49, 0x4a, 0x51, 0x49, 0x75, 0x69, 0x37, 0x72,
  0x37, 0x34, 0x76, 0x64, 0x72, 0x63, 0x58, 0x71, 0x58, 0x0a, 0x44, 0x7a,
  0x68, 0x6d, 0x4c, 0x66, 0x67, 0x6a, 0x67, 0x4c, 0x77, 0x33, 0x2b, 0x55,
  0x79, 0x69, 0x59, 0x74, 0x44, 0x54, 0x76, 0x63, 0x78, 0x65, 0x7a, 0x62,
  0x4c, 0x73, 0x76, 0x51, 0x6f, 0x52, 0x6b, 0x74, 0x77, 0x4b, 0x5a, 0x4c,
  0x44, 0x54, 0x42, 0x42, 0x35, 0x76, 0x59, 0x32, 0x78, 0x4b, 0x36, 0x6b,
  0x4f, 0x4f, 0x44, 0x70, 0x7a, 0x50, 0x48, 0x73, 0x4b, 0x67, 0x30, 0x42,
  0x59, 0x77, 0x0a, 0x4d, 0x6b, 0x48, 0x56, 0x56, 0x54, 0x34, 0x79, 0x2f,
  0x4d, 0x59, 0x36, 0x63, 0x63, 0x4b, 0x51, 0x2f, 0x4c, 0x56, 0x74, 0x32,
  0x66, 0x4a, 0x49, 0x74, 0x69, 0x41, 0x71, 0x49, 0x47, 0x32, 0x38, 0x64,
  0x37, 0x31, 0x53, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44,
  0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45,
  0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a
};

static void testCertTrust(SAFE_PROVIDER_FUNCTIONS *funcs, GUID *actionID)
{
    CRYPT_PROVIDER_DATA data = { 0 };
    CRYPT_PROVIDER_SIGSTATE sig_state = { 0 };
    CRYPT_PROVIDER_SGNR sgnr = { sizeof(sgnr), { 0 } };
    HRESULT ret;
    BOOL b;

    if (!CertFreeCertificateChain_p)
    {
        win_skip("CertFreeCertificateChain not found\n");
        return;
    }

    data.pSigState = &sig_state;
    data.padwTrustStepErrors =
     funcs->pfnAlloc(TRUSTERROR_MAX_STEPS * sizeof(DWORD));
    if (!data.padwTrustStepErrors)
    {
        skip("pfnAlloc failed\n");
        return;
    }
    ret = funcs->pfnCertificateTrust(&data);
    ok(ret == S_FALSE, "Expected S_FALSE, got %08x\n", ret);
    ok(data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_CERTPROV] ==
     TRUST_E_NOSIGNATURE, "Expected TRUST_E_NOSIGNATURE, got %08x\n",
     data.padwTrustStepErrors[TRUSTERROR_STEP_FINAL_CERTPROV]);
    b = funcs->pfnAddSgnr2Chain(&data, FALSE, 0, &sgnr);
    if (b)
    {
        PCCERT_CONTEXT cert;

        /* An empty signer "succeeds," even though there's no cert */
        ret = funcs->pfnCertificateTrust(&data);
        ok(ret == S_OK, "Expected S_OK, got %08x\n", ret);
        cert = CertCreateCertificateContext(X509_ASN_ENCODING, selfSignedCert,
         sizeof(selfSignedCert));
        if (cert)
        {
            WINTRUST_DATA wintrust_data = { 0 };

            b = funcs->pfnAddCert2Chain(&data, 0, FALSE, 0, cert);
            ok(b == TRUE, "Expected TRUE, got %d\n", b);

            /* If pWintrustData isn't set, crashes attempting to access
             * pWintrustData->fdwRevocationChecks
             */
            data.pWintrustData = &wintrust_data;
            /* If psPfns isn't set, crashes attempting to access
             * psPfns->pfnCertCheckPolicy
             */
            data.psPfns = (CRYPT_PROVIDER_FUNCTIONS *)funcs;
            ret = funcs->pfnCertificateTrust(&data);
            ok(ret == S_OK, "Expected S_OK, got %08x\n", ret);
            ok(data.csSigners == 1, "Unexpected number of signers %d\n",
             data.csSigners);
            ok(data.pasSigners[0].pChainContext != NULL,
             "Expected a certificate chain\n");
            ok(data.pasSigners[0].csCertChain == 1,
             "Unexpected number of chain elements %d\n",
             data.pasSigners[0].csCertChain);
            /* pasSigners and pasSigners[0].pasCertChain are guaranteed to be
             * initialized, see tests for pfnAddSgnr2Chain and pfnAddCert2Chain
             */
            ok(!data.pasSigners[0].pasCertChain[0].fTrustedRoot,
             "Didn't expect cert to be trusted\n");
            ok(data.pasSigners[0].pasCertChain[0].fSelfSigned,
             "Expected cert to be self-signed\n");
            ok(data.pasSigners[0].pasCertChain[0].dwConfidence ==
             (CERT_CONFIDENCE_SIG | CERT_CONFIDENCE_TIMENEST),
             "Expected CERT_CONFIDENCE_SIG | CERT_CONFIDENCE_TIMENEST, got %08x\n",
             data.pasSigners[0].pasCertChain[0].dwConfidence);
            CertFreeCertificateContext(
             data.pasSigners[0].pasCertChain[0].pCert);
            CertFreeCertificateChain_p(data.pasSigners[0].pChainContext);
            CertFreeCertificateContext(cert);
        }
    }
    funcs->pfnFree(data.padwTrustStepErrors);
}

static void test_provider_funcs(void)
{
    static GUID generic_verify_v2 = WINTRUST_ACTION_GENERIC_VERIFY_V2;
    SAFE_PROVIDER_FUNCTIONS funcs = { sizeof(SAFE_PROVIDER_FUNCTIONS), 0 };
    BOOL ret;

    ret = WintrustLoadFunctionPointers(&generic_verify_v2,
     (CRYPT_PROVIDER_FUNCTIONS *)&funcs);
    if (!ret)
        skip("WintrustLoadFunctionPointers failed\n");
    else
    {
        test_utils(&funcs);
        testInitialize(&funcs, &generic_verify_v2);
        testObjTrust(&funcs, &generic_verify_v2);
        testCertTrust(&funcs, &generic_verify_v2);
    }
}

/* minimal PE file image */
#define VA_START 0x400000
#define FILE_PE_START 0x50
#define NUM_SECTIONS 3
#define FILE_TEXT 0x200
#define RVA_TEXT 0x1000
#define RVA_BSS 0x2000
#define FILE_IDATA 0x400
#define RVA_IDATA 0x3000
#define FILE_TOTAL 0x600
#define RVA_TOTAL 0x4000
#include <pshpack1.h>
struct Imports {
    IMAGE_IMPORT_DESCRIPTOR descriptors[2];
    IMAGE_THUNK_DATA32 original_thunks[2];
    IMAGE_THUNK_DATA32 thunks[2];
    struct __IMPORT_BY_NAME {
        WORD hint;
        char funcname[0x20];
    } ibn;
    char dllname[0x10];
};
#define EXIT_PROCESS (VA_START+RVA_IDATA+FIELD_OFFSET(struct Imports, thunks))

static struct _PeImage {
    IMAGE_DOS_HEADER dos_header;
    char __alignment1[FILE_PE_START - sizeof(IMAGE_DOS_HEADER)];
    IMAGE_NT_HEADERS32 nt_headers;
    IMAGE_SECTION_HEADER sections[NUM_SECTIONS];
    char __alignment2[FILE_TEXT - FILE_PE_START - sizeof(IMAGE_NT_HEADERS32) -
        NUM_SECTIONS * sizeof(IMAGE_SECTION_HEADER)];
    unsigned char text_section[FILE_IDATA-FILE_TEXT];
    struct Imports idata_section;
    char __alignment3[FILE_TOTAL-FILE_IDATA-sizeof(struct Imports)];
} bin = {
    /* dos header */
    {IMAGE_DOS_SIGNATURE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, {0}, 0, 0, {0}, FILE_PE_START},
    /* alignment before PE header */
    {0},
    /* nt headers */
    {IMAGE_NT_SIGNATURE,
        /* basic headers - 3 sections, no symbols, EXE file */
        {IMAGE_FILE_MACHINE_I386, NUM_SECTIONS, 0, 0, 0, sizeof(IMAGE_OPTIONAL_HEADER32),
            IMAGE_FILE_32BIT_MACHINE | IMAGE_FILE_EXECUTABLE_IMAGE},
        /* optional header */
        {IMAGE_NT_OPTIONAL_HDR32_MAGIC, 4, 0, FILE_IDATA-FILE_TEXT,
            FILE_TOTAL-FILE_IDATA + FILE_IDATA-FILE_TEXT, 0x400,
            RVA_TEXT, RVA_TEXT, RVA_BSS, VA_START, 0x1000, 0x200, 4, 0, 1, 0, 4, 0, 0,
            RVA_TOTAL, FILE_TEXT, 0, IMAGE_SUBSYSTEM_WINDOWS_GUI, 0,
            0x200000, 0x1000, 0x100000, 0x1000, 0, 0x10,
            {{0, 0},
             {RVA_IDATA, sizeof(struct Imports)}
            }
        }
    },
    /* sections */
    {
        {".text", {0x100}, RVA_TEXT, FILE_IDATA-FILE_TEXT, FILE_TEXT,
            0, 0, 0, 0, IMAGE_SCN_CNT_CODE | IMAGE_SCN_MEM_EXECUTE | IMAGE_SCN_MEM_READ},
        {".bss", {0x400}, RVA_BSS, 0, 0, 0, 0, 0, 0,
            IMAGE_SCN_CNT_UNINITIALIZED_DATA | IMAGE_SCN_MEM_READ | IMAGE_SCN_MEM_WRITE},
        {".idata", {sizeof(struct Imports)}, RVA_IDATA, FILE_TOTAL-FILE_IDATA, FILE_IDATA, 0,
            0, 0, 0, IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_MEM_READ | IMAGE_SCN_MEM_WRITE}
    },
    /* alignment before first section */
    {0},
    /* .text section */
    {
        0x31, 0xC0, /* xor eax, eax */
        0xFF, 0x25, EXIT_PROCESS&0xFF, (EXIT_PROCESS>>8)&0xFF, (EXIT_PROCESS>>16)&0xFF,
            (EXIT_PROCESS>>24)&0xFF, /* jmp ExitProcess */
        0
    },
    /* .idata section */
    {
        {
            {{RVA_IDATA + FIELD_OFFSET(struct Imports, original_thunks)}, 0, 0,
            RVA_IDATA + FIELD_OFFSET(struct Imports, dllname),
            RVA_IDATA + FIELD_OFFSET(struct Imports, thunks)
            },
            {{0}, 0, 0, 0, 0}
        },
        {{{RVA_IDATA+FIELD_OFFSET(struct Imports, ibn)}}, {{0}}},
        {{{RVA_IDATA+FIELD_OFFSET(struct Imports, ibn)}}, {{0}}},
        {0,"ExitProcess"},
        "KERNEL32.DLL"
    },
    /* final alignment */
    {0}
};
#include <poppack.h>

static void test_sip_create_indirect_data(void)
{
    static GUID unknown = { 0xC689AAB8, 0x8E78, 0x11D0, { 0x8C,0x47,
     0x00,0xC0,0x4F,0xC2,0x95,0xEE } };
    static char oid_sha1[] = szOID_OIWSEC_sha1;
    BOOL ret;
    SIP_SUBJECTINFO subjinfo = { 0 };
    WCHAR temp_file[MAX_PATH];
    HANDLE file;
    DWORD count;

    if (!CryptSIPCreateIndirectData_p)
    {
        skip("Missing CryptSIPCreateIndirectData\n");
        return;
    }
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(NULL, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(&subjinfo, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
    subjinfo.cbSize = sizeof(subjinfo);
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(&subjinfo, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
    file = create_temp_file(temp_file);
    if (file == INVALID_HANDLE_VALUE)
    {
        skip("couldn't create temp file\n");
        return;
    }
    WriteFile(file, &bin, sizeof(bin), &count, NULL);
    FlushFileBuffers(file);

    subjinfo.hFile = file;
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(&subjinfo, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
    subjinfo.pgSubjectType = &unknown;
    SetLastError(0xdeadbeef);
    ret = CryptSIPCreateIndirectData_p(&subjinfo, NULL, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
       "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
    subjinfo.DigestAlgorithm.pszObjId = oid_sha1;
    count = 0xdeadbeef;
    ret = CryptSIPCreateIndirectData_p(&subjinfo, &count, NULL);
    todo_wine
    ok(ret, "CryptSIPCreateIndirectData failed: %d\n", GetLastError());
    ok(count, "expected a positive count\n");
    if (ret)
    {
        SIP_INDIRECT_DATA *indirect = HeapAlloc(GetProcessHeap(), 0, count);

        count = 256;
        ret = CryptSIPCreateIndirectData_p(&subjinfo, &count, indirect);
        ok(ret, "CryptSIPCreateIndirectData failed: %d\n", GetLastError());
        /* If the count is larger than needed, it's unmodified */
        ok(count == 256, "unexpected count %d\n", count);
        ok(!strcmp(indirect->Data.pszObjId, SPC_PE_IMAGE_DATA_OBJID),
           "unexpected data oid %s\n",
           indirect->Data.pszObjId);
        ok(!strcmp(indirect->DigestAlgorithm.pszObjId, oid_sha1),
           "unexpected digest algorithm oid %s\n",
           indirect->DigestAlgorithm.pszObjId);
        ok(indirect->Digest.cbData == 20, "unexpected hash size %d\n",
           indirect->Digest.cbData);
        if (indirect->Digest.cbData == 20)
        {
            const BYTE hash[20] = {
                0x8a,0xd5,0x45,0x53,0x3d,0x67,0xdf,0x2f,0x78,0xe0,
                0x55,0x0a,0xe0,0xd9,0x7a,0x28,0x3e,0xbf,0x45,0x2b };

            ok(!memcmp(indirect->Digest.pbData, hash, 20),
               "unexpected value\n");
        }

        HeapFree(GetProcessHeap(), 0, indirect);
    }
    CloseHandle(file);
    DeleteFileW(temp_file);
}

static void test_wintrust(void)
{
    static GUID generic_action_v2 = WINTRUST_ACTION_GENERIC_VERIFY_V2;
    WINTRUST_DATA wtd;
    WINTRUST_FILE_INFO file;
    LONG r;
    HRESULT hr;
    WCHAR pathW[MAX_PATH];

    memset(&wtd, 0, sizeof(wtd));
    wtd.cbStruct = sizeof(wtd);
    wtd.dwUIChoice = WTD_UI_NONE;
    wtd.fdwRevocationChecks = WTD_REVOKE_WHOLECHAIN;
    wtd.dwUnionChoice = WTD_CHOICE_FILE;
    U(wtd).pFile = &file;
    wtd.dwStateAction = WTD_STATEACTION_VERIFY;
    memset(&file, 0, sizeof(file));
    file.cbStruct = sizeof(file);
    file.pcwszFilePath = pathW;
    /* Test with an empty file */
    file.hFile = create_temp_file(pathW);
    r = WinVerifyTrust(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(r == TRUST_E_SUBJECT_FORM_UNKNOWN,
     "expected TRUST_E_SUBJECT_FORM_UNKNOWN, got %08x\n", r);
    CloseHandle(file.hFile);
    DeleteFileW(pathW);
    file.hFile = NULL;
    /* Test with a known file path, which we expect not have a signature */
    getNotepadPath(pathW, MAX_PATH);
    r = WinVerifyTrust(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(r == TRUST_E_NOSIGNATURE || r == CRYPT_E_FILE_ERROR,
     "expected TRUST_E_NOSIGNATURE or CRYPT_E_FILE_ERROR, got %08x\n", r);
    wtd.dwStateAction = WTD_STATEACTION_CLOSE;
    r = WinVerifyTrust(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(r == S_OK, "WinVerifyTrust failed: %08x\n", r);
    wtd.dwStateAction = WTD_STATEACTION_VERIFY;
    hr = WinVerifyTrustEx(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(hr == TRUST_E_NOSIGNATURE || hr == CRYPT_E_FILE_ERROR,
     "expected TRUST_E_NOSIGNATURE or CRYPT_E_FILE_ERROR, got %08x\n", hr);
    wtd.dwStateAction = WTD_STATEACTION_CLOSE;
    r = WinVerifyTrust(INVALID_HANDLE_VALUE, &generic_action_v2, &wtd);
    ok(r == S_OK, "WinVerifyTrust failed: %08x\n", r);
}

/* Self-signed .exe, built with tcc, signed with signtool
 * (and a certificate generated on a self-signed CA).
 *
 * small.c:
 * int _start()
 * {
 *     return 0;
 * }
 *
 * tcc -nostdlib small.c
 * signtool sign /v /f codesign.pfx small.exe
 */
static const BYTE SelfSignedFile32[] =
{
    0x4D,0x5A,0x90,0x00,0x03,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0xFF,0xFF,0x00,0x00,0xB8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x0E,0x1F,0xBA,0x0E,0x00,0xB4,0x09,0xCD,
    0x21,0xB8,0x01,0x4C,0xCD,0x21,0x54,0x68,0x69,0x73,0x20,0x70,0x72,0x6F,0x67,0x72,0x61,0x6D,0x20,0x63,0x61,0x6E,0x6E,0x6F,
    0x74,0x20,0x62,0x65,0x20,0x72,0x75,0x6E,0x20,0x69,0x6E,0x20,0x44,0x4F,0x53,0x20,0x6D,0x6F,0x64,0x65,0x2E,0x0D,0x0D,0x0A,
    0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0x45,0x00,0x00,0x4C,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0xE0,0x00,0x0F,0x03,0x0B,0x01,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x10,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x10,0x00,0x00,0x00,0x02,0x00,0x00,
    0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x02,0x00,0x00,
    0xE7,0x0C,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x10,0x00,0x00,
    0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x68,0x05,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x2E,0x74,0x65,0x78,0x74,0x00,0x00,0x00,
    0x18,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x55,0x89,0xE5,0x81,0xEC,0x00,0x00,0x00,0x00,0x90,0xB8,0x00,0x00,0x00,0x00,0xE9,
    0x00,0x00,0x00,0x00,0xC9,0xC3,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x68,0x05,0x00,0x00,0x00,0x02,0x02,0x00,
    /* Start of the signature overlay */
    0x30,0x82,0x05,0x5A,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x07,0x02,0xA0,0x82,0x05,0x4B,0x30,0x82,0x05,0x47,0x02,
    0x01,0x01,0x31,0x0B,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x30,0x4C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,
    0x82,0x37,0x02,0x01,0x04,0xA0,0x3E,0x30,0x3C,0x30,0x17,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0F,0x30,
    0x09,0x03,0x01,0x00,0xA0,0x04,0xA2,0x02,0x80,0x00,0x30,0x21,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x04,
    0x14,0xA0,0x95,0xDE,0xBD,0x1A,0xB7,0x86,0xAF,0x50,0x63,0xD8,0x8F,0x90,0xD5,0x49,0x96,0x4E,0x44,0xF0,0x71,0xA0,0x82,0x03,
    0x1D,0x30,0x82,0x03,0x19,0x30,0x82,0x02,0x01,0xA0,0x03,0x02,0x01,0x02,0x02,0x10,0x96,0x53,0x2C,0xC9,0x23,0x56,0x8A,0x87,
    0x42,0x30,0x3E,0xD5,0x8D,0x72,0xD5,0x25,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,
    0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,
    0x30,0x1E,0x17,0x0D,0x31,0x36,0x30,0x33,0x30,0x33,0x32,0x30,0x32,0x37,0x30,0x37,0x5A,0x17,0x0D,0x34,0x39,0x31,0x32,0x33,
    0x31,0x32,0x33,0x30,0x30,0x30,0x30,0x5A,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x43,0x6F,0x64,
    0x65,0x53,0x69,0x67,0x6E,0x54,0x65,0x73,0x74,0x30,0x82,0x01,0x22,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,
    0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0F,0x00,0x30,0x82,0x01,0x0A,0x02,0x82,0x01,0x01,0x00,0xB2,0xC9,0x91,0x98,0x8C,0xDC,
    0x80,0xBC,0x16,0xBF,0xC1,0x04,0x77,0x90,0xC0,0xFD,0x8C,0xBA,0x68,0x26,0xAC,0xB7,0x20,0x68,0x41,0xED,0xC3,0x9C,0x47,0x7C,
    0x36,0xC2,0x7B,0xE1,0x5E,0xFD,0xA9,0x99,0xF4,0x29,0x36,0x86,0x93,0x40,0x55,0x53,0x65,0x79,0xBC,0x9F,0x8F,0x6E,0x2B,0x05,
    0x84,0xE1,0xFD,0xD2,0xEF,0xEA,0x89,0x8C,0xEC,0xF9,0x55,0xF0,0x2C,0xE5,0xA7,0x29,0xF9,0x7E,0x50,0xDC,0x9C,0xA1,0x23,0xA5,
    0xD9,0x78,0xA1,0xE7,0x7C,0xD7,0x04,0x4F,0x11,0xAC,0x9F,0x4A,0x47,0xA1,0x1E,0xD5,0x9E,0xE7,0x5B,0xB5,0x8C,0x9C,0x67,0x7A,
    0xD0,0xF8,0x54,0xD1,0x64,0x7F,0x39,0x48,0xB6,0xCF,0x2F,0x26,0x7D,0x7B,0x13,0x2B,0xC2,0x8F,0xA6,0x3F,0x42,0x71,0x95,0x3E,
    0x59,0x0F,0x12,0xFA,0xC2,0x70,0x89,0xB7,0xB6,0x10,0x49,0xE0,0x7D,0x4D,0xFC,0x80,0x61,0x53,0x50,0x72,0xFD,0x46,0x35,0x51,
    0x36,0xE6,0x06,0xA9,0x4C,0x0D,0x82,0x15,0xF6,0x5D,0xDE,0xD4,0xDB,0xE7,0x82,0x10,0x40,0xA1,0x47,0x68,0x88,0x0C,0x0A,0x80,
    0xD1,0xE5,0x9A,0x35,0x28,0x82,0x1F,0x0F,0x80,0x5A,0x6E,0x1D,0x22,0x22,0xB3,0xA7,0xA2,0x9E,0x82,0x2D,0xC0,0x7F,0x5A,0xD0,
    0xBA,0xB2,0xCA,0x20,0xE2,0x97,0xE9,0x72,0x41,0xB7,0xD6,0x1A,0x93,0x23,0x97,0xF0,0xA9,0x61,0xD2,0x91,0xBD,0xB6,0x6B,0x95,
    0x12,0x67,0x16,0xAC,0x0A,0xB7,0x55,0x02,0x0D,0xA5,0xAD,0x17,0x95,0x77,0xF9,0x96,0x03,0x41,0xD3,0xE1,0x61,0x68,0xBB,0x0A,
    0xB5,0xC4,0xEE,0x70,0x40,0x08,0x05,0xC4,0xF1,0x5D,0x02,0x03,0x01,0x00,0x01,0xA3,0x61,0x30,0x5F,0x30,0x13,0x06,0x03,0x55,
    0x1D,0x25,0x04,0x0C,0x30,0x0A,0x06,0x08,0x2B,0x06,0x01,0x05,0x05,0x07,0x03,0x03,0x30,0x48,0x06,0x03,0x55,0x1D,0x01,0x04,
    0x41,0x30,0x3F,0x80,0x10,0x35,0x40,0x67,0x8F,0x7D,0x03,0x1B,0x76,0x52,0x62,0x2D,0xF5,0x21,0xF6,0x7C,0xBC,0xA1,0x19,0x30,
    0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,
    0x82,0x10,0xA0,0x4B,0xEB,0xAC,0xFA,0x08,0xF2,0x8B,0x47,0xD2,0xB3,0x54,0x60,0x6C,0xE6,0x29,0x30,0x0D,0x06,0x09,0x2A,0x86,
    0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x5F,0x8C,0x7F,0xDA,0x1D,0x21,0x7A,0x15,0xD8,0x20,
    0x04,0x53,0x7F,0x44,0x6D,0x7B,0x57,0xBE,0x7F,0x86,0x77,0x58,0xC4,0xD4,0x80,0xC7,0x2E,0x64,0x9B,0x44,0xC5,0x2D,0x6D,0xDB,
    0x35,0x5A,0xFE,0xA4,0xD8,0x66,0x9B,0xF7,0x6E,0xFC,0xEF,0x52,0x7B,0xC5,0x16,0xE6,0xA3,0x7D,0x59,0xB7,0x31,0x28,0xEB,0xB5,
    0x45,0xC9,0xB1,0xD1,0x08,0x67,0xC6,0x37,0xE7,0xD7,0x2A,0xE6,0x1F,0xD9,0x6A,0xE5,0x04,0xDF,0x6A,0x9D,0x91,0xFA,0x41,0xBD,
    0x2A,0x50,0xEA,0x99,0x24,0xA9,0x0F,0x2B,0x50,0x51,0x5F,0xD9,0x0B,0x89,0x1B,0xCB,0xDB,0x88,0xE8,0xEC,0x87,0xB0,0x16,0xCC,
    0x43,0xEE,0x5A,0xBD,0x57,0xE2,0x46,0xA7,0x56,0x54,0x23,0x32,0x8A,0xFB,0x25,0x51,0x39,0x38,0xE6,0x87,0xF5,0x73,0x63,0xD0,
    0x5B,0xC7,0x3F,0xFD,0x04,0x75,0x74,0x4C,0x3D,0xB5,0x31,0x22,0x7D,0xF1,0x8D,0xB4,0xE0,0xAA,0xE1,0xFF,0x8F,0xDD,0xB8,0x04,
    0x6A,0x31,0xEE,0x30,0x2D,0x6E,0x74,0x0F,0x37,0x71,0x77,0x2B,0xB8,0x9E,0x62,0x47,0x00,0x9C,0xA5,0x82,0x2B,0x9F,0x24,0x67,
    0x50,0x86,0x8B,0xC9,0x36,0x81,0xEB,0x44,0xC2,0xF1,0x91,0xA6,0x84,0x75,0x15,0x8F,0x22,0xDE,0xAC,0xB5,0x16,0xE3,0x96,0x74,
    0x72,0x2F,0x15,0xD5,0xFB,0x01,0x22,0xC4,0x24,0xEE,0x3D,0xDF,0x9E,0xA9,0x0A,0x5B,0x16,0x21,0xE8,0x4A,0x8C,0x7E,0x3A,0x9C,
    0x22,0xA0,0x49,0x60,0x97,0x1B,0x3E,0x2D,0x80,0x91,0xDB,0xF7,0x78,0x38,0x76,0x78,0x0C,0xE3,0xD4,0x27,0x77,0x69,0x96,0xE6,
    0x41,0xC7,0x2E,0xE9,0x61,0xD6,0x31,0x82,0x01,0xC4,0x30,0x82,0x01,0xC0,0x02,0x01,0x01,0x30,0x2B,0x30,0x17,0x31,0x15,0x30,
    0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,0x02,0x10,0x96,0x53,
    0x2C,0xC9,0x23,0x56,0x8A,0x87,0x42,0x30,0x3E,0xD5,0x8D,0x72,0xD5,0x25,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,
    0x00,0xA0,0x70,0x30,0x10,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0C,0x31,0x02,0x30,0x00,0x30,0x19,0x06,
    0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x09,0x03,0x31,0x0C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,
    0x30,0x1C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0B,0x31,0x0E,0x30,0x0C,0x06,0x0A,0x2B,0x06,0x01,0x04,
    0x01,0x82,0x37,0x02,0x01,0x15,0x30,0x23,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x09,0x04,0x31,0x16,0x04,0x14,0x3D,
    0x08,0xC8,0xA3,0xEE,0x05,0x1A,0x61,0xD9,0xFE,0x1A,0x63,0xC0,0x8A,0x6E,0x9D,0xF9,0xC3,0x13,0x98,0x30,0x0D,0x06,0x09,0x2A,
    0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x01,0x00,0x90,0xF9,0xC0,0x7F,0x1D,0x70,0x8C,0x04,0x22,0x82,
    0xB6,0x2D,0x48,0xBF,0x30,0x51,0x29,0xF8,0xE3,0x11,0x39,0xE0,0x64,0x23,0x72,0xE2,0x4C,0x09,0x9F,0x39,0xF2,0x6F,0xDD,0xB9,
    0x5A,0x3D,0xEF,0xEB,0xBE,0xEC,0x3B,0xE6,0x58,0x4C,0xC9,0x4F,0xED,0xCB,0x6E,0x9D,0x67,0x8E,0x89,0x92,0x40,0x39,0xA2,0x5F,
    0xF9,0xEF,0xD3,0xF5,0x24,0x27,0x8D,0xF7,0x3C,0x92,0x66,0x56,0xC8,0x2B,0xEA,0x04,0xA1,0x0E,0xDA,0x89,0x30,0xA7,0x01,0xD8,
    0x0B,0xF8,0xFD,0x99,0xB6,0xC0,0x38,0xB0,0x21,0x50,0x3A,0x86,0x01,0xD0,0xF3,0x86,0x72,0xE3,0x5A,0xBB,0x2A,0x6E,0xBD,0xFB,
    0x22,0xF9,0x42,0xD3,0x04,0xFE,0x8D,0xD8,0x79,0xD1,0xEE,0x61,0xC6,0x48,0x04,0x99,0x9A,0xA2,0x73,0xE5,0xFB,0x24,0x10,0xD5,
    0x6B,0x71,0x80,0x0E,0x09,0xEA,0x85,0x9A,0xBD,0xBB,0xDE,0x99,0x5D,0xA3,0x18,0x4D,0xED,0x20,0x73,0x3E,0x32,0xEF,0x2C,0xAC,
    0x5A,0x83,0x87,0x1F,0x7F,0x19,0x61,0x35,0x53,0xC1,0xAA,0x89,0x97,0xB3,0xDD,0x8D,0xA8,0x67,0x5B,0xC2,0xE2,0x09,0xB7,0xDD,
    0x6A,0xCB,0xD5,0xBF,0xD6,0x08,0xE2,0x23,0x1A,0x41,0x9D,0xD5,0x6A,0x6B,0x8D,0x3C,0x29,0x1B,0xF1,0x3F,0x4E,0x4A,0x8F,0x29,
    0x33,0xF9,0x1C,0x60,0xA0,0x92,0x7E,0x4F,0x35,0xB8,0xDD,0xEB,0xD1,0x68,0x1A,0x9D,0xA2,0xA6,0x97,0x1F,0x5F,0xC6,0x2C,0xFB,
    0xCA,0xDF,0xF7,0x95,0x33,0x95,0xD4,0x79,0x5C,0x73,0x87,0x49,0x1F,0x8C,0x6E,0xCE,0x3E,0x6D,0x3D,0x2B,0x6B,0xD7,0x66,0xE9,
    0x88,0x6F,0xF2,0x83,0xB9,0x9B,0x00,0x00
};

static const BYTE SelfSignedFile64[] =
{
    0x4D,0x5A,0x90,0x00,0x03,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0xFF,0xFF,0x00,0x00,0xB8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x0E,0x1F,0xBA,0x0E,0x00,0xB4,0x09,0xCD,
    0x21,0xB8,0x01,0x4C,0xCD,0x21,0x54,0x68,0x69,0x73,0x20,0x70,0x72,0x6F,0x67,0x72,0x61,0x6D,0x20,0x63,0x61,0x6E,0x6E,0x6F,
    0x74,0x20,0x62,0x65,0x20,0x72,0x75,0x6E,0x20,0x69,0x6E,0x20,0x44,0x4F,0x53,0x20,0x6D,0x6F,0x64,0x65,0x2E,0x0D,0x0D,0x0A,
    0x24,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x50,0x45,0x00,0x00,0x64,0x86,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0xF0,0x00,0x2F,0x02,0x0B,0x02,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x10,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x02,0x00,0x00,
    0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x02,0x00,0x00,
    0x02,0xB9,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x20,0x00,0x00,0x0C,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x68,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x2E,0x74,0x65,0x78,0x74,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x10,0x00,0x00,
    0x00,0x02,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x60,
    0x2E,0x70,0x64,0x61,0x74,0x61,0x00,0x00,0x0C,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x02,0x00,0x00,0x00,0x04,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x55,0x48,0x89,0xE5,0x48,0x81,0xEC,0x00,0x00,0x00,0x00,0xB8,0x00,0x00,0x00,0x00,
    0xE9,0x00,0x00,0x00,0x00,0xC9,0xC3,0x00,0x01,0x04,0x02,0x05,0x04,0x03,0x01,0x50,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0B,0x10,0x00,0x00,0x17,0x10,0x00,0x00,
    0x18,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    /* Start of the signature overlay */
    0x68,0x05,0x00,0x00,0x00,0x02,0x02,0x00,0x30,0x82,0x05,0x5A,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x07,0x02,0xA0,
    0x82,0x05,0x4B,0x30,0x82,0x05,0x47,0x02,0x01,0x01,0x31,0x0B,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x30,
    0x4C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,0xA0,0x3E,0x30,0x3C,0x30,0x17,0x06,0x0A,0x2B,0x06,0x01,
    0x04,0x01,0x82,0x37,0x02,0x01,0x0F,0x30,0x09,0x03,0x01,0x00,0xA0,0x04,0xA2,0x02,0x80,0x00,0x30,0x21,0x30,0x09,0x06,0x05,
    0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x04,0x14,0xCA,0x7C,0x10,0xFB,0x5A,0x96,0x6D,0x69,0xEF,0x26,0x30,0x1A,0xE9,0xC7,0x22,
    0x19,0xEB,0x6E,0x17,0x07,0xA0,0x82,0x03,0x1D,0x30,0x82,0x03,0x19,0x30,0x82,0x02,0x01,0xA0,0x03,0x02,0x01,0x02,0x02,0x10,
    0x96,0x53,0x2C,0xC9,0x23,0x56,0x8A,0x87,0x42,0x30,0x3E,0xD5,0x8D,0x72,0xD5,0x25,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,
    0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,
    0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,0x30,0x1E,0x17,0x0D,0x31,0x36,0x30,0x33,0x30,0x33,0x32,0x30,0x32,0x37,0x30,0x37,
    0x5A,0x17,0x0D,0x34,0x39,0x31,0x32,0x33,0x31,0x32,0x33,0x30,0x30,0x30,0x30,0x5A,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,
    0x55,0x04,0x03,0x13,0x0C,0x43,0x6F,0x64,0x65,0x53,0x69,0x67,0x6E,0x54,0x65,0x73,0x74,0x30,0x82,0x01,0x22,0x30,0x0D,0x06,
    0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0F,0x00,0x30,0x82,0x01,0x0A,0x02,0x82,0x01,
    0x01,0x00,0xB2,0xC9,0x91,0x98,0x8C,0xDC,0x80,0xBC,0x16,0xBF,0xC1,0x04,0x77,0x90,0xC0,0xFD,0x8C,0xBA,0x68,0x26,0xAC,0xB7,
    0x20,0x68,0x41,0xED,0xC3,0x9C,0x47,0x7C,0x36,0xC2,0x7B,0xE1,0x5E,0xFD,0xA9,0x99,0xF4,0x29,0x36,0x86,0x93,0x40,0x55,0x53,
    0x65,0x79,0xBC,0x9F,0x8F,0x6E,0x2B,0x05,0x84,0xE1,0xFD,0xD2,0xEF,0xEA,0x89,0x8C,0xEC,0xF9,0x55,0xF0,0x2C,0xE5,0xA7,0x29,
    0xF9,0x7E,0x50,0xDC,0x9C,0xA1,0x23,0xA5,0xD9,0x78,0xA1,0xE7,0x7C,0xD7,0x04,0x4F,0x11,0xAC,0x9F,0x4A,0x47,0xA1,0x1E,0xD5,
    0x9E,0xE7,0x5B,0xB5,0x8C,0x9C,0x67,0x7A,0xD0,0xF8,0x54,0xD1,0x64,0x7F,0x39,0x48,0xB6,0xCF,0x2F,0x26,0x7D,0x7B,0x13,0x2B,
    0xC2,0x8F,0xA6,0x3F,0x42,0x71,0x95,0x3E,0x59,0x0F,0x12,0xFA,0xC2,0x70,0x89,0xB7,0xB6,0x10,0x49,0xE0,0x7D,0x4D,0xFC,0x80,
    0x61,0x53,0x50,0x72,0xFD,0x46,0x35,0x51,0x36,0xE6,0x06,0xA9,0x4C,0x0D,0x82,0x15,0xF6,0x5D,0xDE,0xD4,0xDB,0xE7,0x82,0x10,
    0x40,0xA1,0x47,0x68,0x88,0x0C,0x0A,0x80,0xD1,0xE5,0x9A,0x35,0x28,0x82,0x1F,0x0F,0x80,0x5A,0x6E,0x1D,0x22,0x22,0xB3,0xA7,
    0xA2,0x9E,0x82,0x2D,0xC0,0x7F,0x5A,0xD0,0xBA,0xB2,0xCA,0x20,0xE2,0x97,0xE9,0x72,0x41,0xB7,0xD6,0x1A,0x93,0x23,0x97,0xF0,
    0xA9,0x61,0xD2,0x91,0xBD,0xB6,0x6B,0x95,0x12,0x67,0x16,0xAC,0x0A,0xB7,0x55,0x02,0x0D,0xA5,0xAD,0x17,0x95,0x77,0xF9,0x96,
    0x03,0x41,0xD3,0xE1,0x61,0x68,0xBB,0x0A,0xB5,0xC4,0xEE,0x70,0x40,0x08,0x05,0xC4,0xF1,0x5D,0x02,0x03,0x01,0x00,0x01,0xA3,
    0x61,0x30,0x5F,0x30,0x13,0x06,0x03,0x55,0x1D,0x25,0x04,0x0C,0x30,0x0A,0x06,0x08,0x2B,0x06,0x01,0x05,0x05,0x07,0x03,0x03,
    0x30,0x48,0x06,0x03,0x55,0x1D,0x01,0x04,0x41,0x30,0x3F,0x80,0x10,0x35,0x40,0x67,0x8F,0x7D,0x03,0x1B,0x76,0x52,0x62,0x2D,
    0xF5,0x21,0xF6,0x7C,0xBC,0xA1,0x19,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,
    0x20,0x43,0x41,0x20,0x52,0x6F,0x6F,0x74,0x82,0x10,0xA0,0x4B,0xEB,0xAC,0xFA,0x08,0xF2,0x8B,0x47,0xD2,0xB3,0x54,0x60,0x6C,
    0xE6,0x29,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x0B,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x5F,0x8C,
    0x7F,0xDA,0x1D,0x21,0x7A,0x15,0xD8,0x20,0x04,0x53,0x7F,0x44,0x6D,0x7B,0x57,0xBE,0x7F,0x86,0x77,0x58,0xC4,0xD4,0x80,0xC7,
    0x2E,0x64,0x9B,0x44,0xC5,0x2D,0x6D,0xDB,0x35,0x5A,0xFE,0xA4,0xD8,0x66,0x9B,0xF7,0x6E,0xFC,0xEF,0x52,0x7B,0xC5,0x16,0xE6,
    0xA3,0x7D,0x59,0xB7,0x31,0x28,0xEB,0xB5,0x45,0xC9,0xB1,0xD1,0x08,0x67,0xC6,0x37,0xE7,0xD7,0x2A,0xE6,0x1F,0xD9,0x6A,0xE5,
    0x04,0xDF,0x6A,0x9D,0x91,0xFA,0x41,0xBD,0x2A,0x50,0xEA,0x99,0x24,0xA9,0x0F,0x2B,0x50,0x51,0x5F,0xD9,0x0B,0x89,0x1B,0xCB,
    0xDB,0x88,0xE8,0xEC,0x87,0xB0,0x16,0xCC,0x43,0xEE,0x5A,0xBD,0x57,0xE2,0x46,0xA7,0x56,0x54,0x23,0x32,0x8A,0xFB,0x25,0x51,
    0x39,0x38,0xE6,0x87,0xF5,0x73,0x63,0xD0,0x5B,0xC7,0x3F,0xFD,0x04,0x75,0x74,0x4C,0x3D,0xB5,0x31,0x22,0x7D,0xF1,0x8D,0xB4,
    0xE0,0xAA,0xE1,0xFF,0x8F,0xDD,0xB8,0x04,0x6A,0x31,0xEE,0x30,0x2D,0x6E,0x74,0x0F,0x37,0x71,0x77,0x2B,0xB8,0x9E,0x62,0x47,
    0x00,0x9C,0xA5,0x82,0x2B,0x9F,0x24,0x67,0x50,0x86,0x8B,0xC9,0x36,0x81,0xEB,0x44,0xC2,0xF1,0x91,0xA6,0x84,0x75,0x15,0x8F,
    0x22,0xDE,0xAC,0xB5,0x16,0xE3,0x96,0x74,0x72,0x2F,0x15,0xD5,0xFB,0x01,0x22,0xC4,0x24,0xEE,0x3D,0xDF,0x9E,0xA9,0x0A,0x5B,
    0x16,0x21,0xE8,0x4A,0x8C,0x7E,0x3A,0x9C,0x22,0xA0,0x49,0x60,0x97,0x1B,0x3E,0x2D,0x80,0x91,0xDB,0xF7,0x78,0x38,0x76,0x78,
    0x0C,0xE3,0xD4,0x27,0x77,0x69,0x96,0xE6,0x41,0xC7,0x2E,0xE9,0x61,0xD6,0x31,0x82,0x01,0xC4,0x30,0x82,0x01,0xC0,0x02,0x01,
    0x01,0x30,0x2B,0x30,0x17,0x31,0x15,0x30,0x13,0x06,0x03,0x55,0x04,0x03,0x13,0x0C,0x54,0x65,0x73,0x74,0x20,0x43,0x41,0x20,
    0x52,0x6F,0x6F,0x74,0x02,0x10,0x96,0x53,0x2C,0xC9,0x23,0x56,0x8A,0x87,0x42,0x30,0x3E,0xD5,0x8D,0x72,0xD5,0x25,0x30,0x09,
    0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0xA0,0x70,0x30,0x10,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,
    0x0C,0x31,0x02,0x30,0x00,0x30,0x19,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x09,0x03,0x31,0x0C,0x06,0x0A,0x2B,0x06,
    0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x04,0x30,0x1C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x0B,0x31,0x0E,
    0x30,0x0C,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x82,0x37,0x02,0x01,0x15,0x30,0x23,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,
    0x01,0x09,0x04,0x31,0x16,0x04,0x14,0x0C,0xEC,0x76,0xF2,0x3F,0xE4,0x6F,0xEB,0xFF,0x00,0xDA,0x95,0xE7,0x8B,0x64,0xBC,0x55,
    0xBA,0xF0,0xEA,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x01,0x00,0x05,0x22,
    0xD1,0xB3,0x85,0x09,0x46,0x99,0x77,0x69,0xC8,0xD2,0x0C,0xFC,0x8D,0xF4,0x01,0xD4,0x5B,0xF0,0xB4,0x13,0x63,0xAF,0x24,0x0E,
    0x6C,0x1B,0x14,0xCF,0xA8,0x9A,0xEC,0x7E,0xF2,0x60,0xED,0x6C,0x39,0x4D,0x7A,0x73,0x9C,0x9F,0x24,0x46,0xE2,0xEA,0xFA,0x70,
    0xB4,0xAC,0xFC,0x38,0x90,0xF2,0x4F,0x70,0xCC,0x00,0xD1,0x2B,0xB6,0xFB,0xCD,0x7F,0xFC,0xCB,0x35,0xA9,0xA6,0x76,0x37,0xD6,
    0x08,0x82,0x99,0x4C,0x47,0xD7,0x4E,0xB5,0xDE,0xCA,0x4E,0xED,0x71,0x48,0xD4,0x84,0xE1,0x30,0x10,0x33,0x7F,0x84,0xEE,0x2F,
    0x44,0x99,0xE4,0x26,0x27,0xB5,0xB8,0xC1,0xA1,0x40,0x6B,0x87,0x04,0x95,0xC3,0xF0,0xFF,0x25,0x97,0xFD,0xDB,0x9C,0x67,0x80,
    0x39,0x97,0x72,0x75,0x07,0x92,0xA5,0x08,0x19,0x5B,0xD3,0xC9,0x5E,0xC4,0x7B,0xA9,0x04,0x02,0x63,0xCC,0xC5,0x92,0xF6,0xE9,
    0xD6,0xB0,0xA8,0xF9,0xD0,0x9F,0x3F,0xBC,0x86,0x77,0x1E,0x12,0x9A,0x9A,0x9B,0x05,0x77,0x39,0x42,0x01,0xB7,0x23,0xF0,0x78,
    0x4F,0x52,0x6D,0x1B,0x9F,0xBA,0x29,0xEC,0x90,0xA9,0x1E,0x1E,0x5C,0xA9,0x28,0xA0,0x0B,0x09,0xDC,0x99,0x82,0xE3,0x34,0xBB,
    0x5C,0x66,0x8E,0x54,0x95,0x4B,0x65,0x95,0xCD,0x87,0x72,0x74,0xCD,0x3B,0x5C,0x72,0xBB,0x61,0x6A,0x98,0x44,0x9C,0xB0,0x2A,
    0xE7,0xB0,0xA6,0x2B,0xDA,0x47,0x5C,0x75,0x36,0xB5,0x90,0x8E,0x82,0x47,0xCD,0x3F,0x4B,0xD0,0xFB,0x8E,0x17,0x6B,0x40,0x57,
    0x9C,0x68,0x1A,0x5D,0x92,0xCD,0xD0,0x5F,0x02,0xA1,0x2C,0xD9,0x56,0x20,0x00,0x00
};

static void call_winverify(WCHAR *pathW, LONG *status, BOOL hash_only)
{
    static GUID WVTPolicyGUID = WINTRUST_ACTION_GENERIC_VERIFY_V2;
    WINTRUST_FILE_INFO file_info = { sizeof(file_info), 0 };
    WINTRUST_DATA data = { sizeof(data), 0 };
    LONG ret;

    file_info.pcwszFilePath     = pathW;

    data.dwUIChoice             = WTD_UI_NONE;
    data.fdwRevocationChecks    = WTD_REVOKE_NONE;
    data.dwUnionChoice          = WTD_CHOICE_FILE;
    data.pFile                  = &file_info;
    data.dwStateAction          = WTD_STATEACTION_VERIFY;
    data.dwProvFlags            = hash_only ? WTD_HASH_ONLY_FLAG : 0;
    *status = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);

    data.dwStateAction = WTD_STATEACTION_CLOSE;
    ret = WinVerifyTrust(NULL, &WVTPolicyGUID, &data);
    ok(ret == S_OK, "WinVerifyTrust failed: %08x\n", ret);
}

static void test_wintrust_digest(void)
{
    static const BYTE Dummy[] = { 0x11,0x22,0x33,0x44 };
    static const struct
    {
        struct { const BYTE *data; DWORD length; } blocks[5];
        struct { LONG status; BOOL todo; } t1;
        struct { LONG status; BOOL todo; } t2;
    }
    tests[] =
    {
        /* 32-bit tests */
        {
            {{ SelfSignedFile32, sizeof(SelfSignedFile32) }},
            { CERT_E_CHAINING, TRUE }, { S_OK, FALSE }
        },
        {
            {{ SelfSignedFile32, sizeof(SelfSignedFile32) },
             { Dummy, sizeof(Dummy) }},
            { TRUST_E_NOSIGNATURE, FALSE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
        {
            {{ Dummy, sizeof(Dummy) },
             { SelfSignedFile32 + sizeof(Dummy), sizeof(SelfSignedFile32) - sizeof(Dummy) }},
            { TRUST_E_SUBJECT_FORM_UNKNOWN, FALSE }, { TRUST_E_NOSIGNATURE, TRUE }
        },
        {
            {{ SelfSignedFile32, 19 },
             { Dummy, sizeof(Dummy) },
             { SelfSignedFile32 + 19 + sizeof(Dummy), sizeof(SelfSignedFile32) - 19 - sizeof(Dummy) }},
            { TRUST_E_BAD_DIGEST, FALSE }, { TRUST_E_NOSIGNATURE, TRUE }
        },
        {
            {{ SelfSignedFile32, sizeof(IMAGE_DOS_HEADER) }},
            { TRUST_E_SUBJECT_FORM_UNKNOWN, TRUE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
        {
            {{ SelfSignedFile32, sizeof(IMAGE_DOS_HEADER) + sizeof(IMAGE_NT_HEADERS32) * 2 }},
            { TRUST_E_NOSIGNATURE, FALSE }, { TRUST_E_NOSIGNATURE, FALSE }
        },

        /* 64-bit tests */
        {
            {{ SelfSignedFile64, sizeof(SelfSignedFile64) }},
            { CERT_E_CHAINING, TRUE }, { S_OK, FALSE }
        },
        {
            {{ SelfSignedFile64, sizeof(SelfSignedFile64) },
             { Dummy, sizeof(Dummy) }},
            { TRUST_E_NOSIGNATURE, FALSE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
        {
            {{ Dummy, sizeof(Dummy) },
             { SelfSignedFile64 + sizeof(Dummy), sizeof(SelfSignedFile64) - sizeof(Dummy) }},
            { TRUST_E_SUBJECT_FORM_UNKNOWN, FALSE }, { TRUST_E_NOSIGNATURE, TRUE }
        },
        {
            {{ SelfSignedFile64, 19 },
             { Dummy, sizeof(Dummy) },
             { SelfSignedFile64 + 19 + sizeof(Dummy), sizeof(SelfSignedFile64) - 19 - sizeof(Dummy) }},
            { TRUST_E_BAD_DIGEST, FALSE }, { TRUST_E_NOSIGNATURE, TRUE }
        },
        {
            {{ SelfSignedFile64, sizeof(IMAGE_DOS_HEADER) }},
            { TRUST_E_SUBJECT_FORM_UNKNOWN, TRUE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
        {
            {{ SelfSignedFile64, sizeof(IMAGE_DOS_HEADER) + sizeof(IMAGE_NT_HEADERS64) * 2 }},
            { TRUST_E_NOSIGNATURE, FALSE }, { TRUST_E_NOSIGNATURE, FALSE }
        },
    };
    WCHAR pathW[MAX_PATH];
    DWORD written;
    HANDLE file;
    LONG status;
    BOOL ret;
    int i, j;

    for (i = 0; i < sizeof(tests)/sizeof(tests[0]); i++)
    {
        file = create_temp_file(pathW);
        ok(file != INVALID_HANDLE_VALUE, "failed to create temporary file\n");

        for (j = 0; tests[i].blocks[j].data; j++)
        {
            ret = WriteFile(file, tests[i].blocks[j].data, tests[i].blocks[j].length, &written, NULL);
            ok(ret && written == tests[i].blocks[j].length, "WriteFile failed with %u\n", GetLastError());
        }

        CloseHandle(file);

        call_winverify(pathW, &status, FALSE);
        todo_wine_if(tests[i].t1.todo)
        ok(status == tests[i].t1.status, "test %d/1: expected %08x, got %08x\n", i, tests[i].t1.status, status);

        call_winverify(pathW, &status, TRUE);
        todo_wine_if(tests[i].t2.todo)
        ok(status == tests[i].t2.status, "test %d/2: expected %08x, got %08x\n", i, tests[i].t2.status, status);

        DeleteFileW(pathW);
    }
}

static void test_get_known_usages(void)
{
    BOOL ret;
    PCCRYPT_OID_INFO *usages;

    if (!pWTHelperGetKnownUsages)
    {
        skip("missing WTHelperGetKnownUsages\n");
        return;
    }
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(0, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
     "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(1, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
     "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(0, &usages);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
     "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
    /* A value of 1 for the first parameter seems to imply the value is
     * allocated
     */
    SetLastError(0xdeadbeef);
    usages = NULL;
    ret = pWTHelperGetKnownUsages(1, &usages);
    ok(ret, "WTHelperGetKnownUsages failed: %d\n", GetLastError());
    ok(usages != NULL, "expected a pointer\n");
    if (ret && usages)
    {
        PCCRYPT_OID_INFO *ptr;

        /* The returned usages are an array of PCCRYPT_OID_INFOs, terminated with a
         * NULL pointer.
         */
        for (ptr = usages; *ptr; ptr++)
        {
            ok((*ptr)->cbSize == sizeof(CRYPT_OID_INFO) ||
             (*ptr)->cbSize == (sizeof(CRYPT_OID_INFO) + 2 * sizeof(LPCWSTR)), /* Vista */
             "unexpected size %d\n", (*ptr)->cbSize);
            /* Each returned usage is in the CRYPT_ENHKEY_USAGE_OID_GROUP_ID group */
            ok((*ptr)->dwGroupId == CRYPT_ENHKEY_USAGE_OID_GROUP_ID,
             "expected group CRYPT_ENHKEY_USAGE_OID_GROUP_ID, got %d\n",
             (*ptr)->dwGroupId);
        }
    }
    /* A value of 2 for the second parameter seems to imply the value is freed
     */
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(2, &usages);
    ok(ret, "WTHelperGetKnownUsages failed: %d\n", GetLastError());
    ok(usages == NULL, "expected pointer to be cleared\n");
    SetLastError(0xdeadbeef);
    usages = NULL;
    ret = pWTHelperGetKnownUsages(2, &usages);
    ok(ret, "WTHelperGetKnownUsages failed: %d\n", GetLastError());
    SetLastError(0xdeadbeef);
    ret = pWTHelperGetKnownUsages(2, NULL);
    ok(!ret && GetLastError() == ERROR_INVALID_PARAMETER,
     "expected ERROR_INVALID_PARAMETER, got %d\n", GetLastError());
}

START_TEST(softpub)
{
    InitFunctionPtrs();
    test_provider_funcs();
    test_sip_create_indirect_data();
    test_wintrust();
    test_wintrust_digest();
    test_get_known_usages();
}
